/*
* KAMATSU'S CLASS LIBRARY
* Version 0.45
*  
* Created by Kamatsu
* 
* Released under the MIT license.
* 
* Copyright (C) 2007 Liam O'Connor-Davis e.a. Only a couple of rights reserved.
* Feel free to use/modify this for your projects, provided this copyright notice
* remains intact.
*
* Also, no warranty. Either expressed or implied, including the implied warranties
* of merchantability or fitness for a particular purpose. 
*
* Enjoy.
*
*/

/*

kclWindowClasses.js: Windowing classes such as MessageWindow.

*/

try {
    LibraryRegistry.abortIfNotInstalled("Spheritype",1.1,false);
}
catch (ex) {
    Abort("Spheritype Not Installed!");
}

LibraryRegistry.abortIfNotInstalled("Kamatsu's Class Library: Declares",0.45, true); //Versions must equate :), hence the strict arg.
LibraryRegistry.abortIfNotInstalled("Kamatsu's Class Library: Wrapper Classes",0.45, true); //Versions must equate :), hence the strict arg.
LibraryRegistry.abortIfNotInstalled("Kamatsu's Class Library: Prototype Extensions",0.45, true); //Versions must equate :), hence the strict arg.
LibraryRegistry.addLibraryData("Kamatsu's Class Library: Window Classes",0.45);

/* OBJECT: Window 
    Parameters (all optional): 
       *  windowstyle - windowstyle for the window object to use
       *  x, y, w, h  - Dimensions

    Non Parametised Properties
       *  margin - Spacing of window. Default 20. 
       *  opacity - Opacity of window. Default 255
       *  visible - whether to draw the window on redraw. Useful for hiding stuff.
       *  animInc - rate of change of animation
       *  animSpeed - fps for animation
       *  introAnim - animation to use for intro
       *  outtroAnim - animation to use for outro
       *  background - Either BACKGROUND_NONE, BACKGROUND_RENDERMAP or an image/surface to render as background       
       *  flipScreen - whether or not to flip the screen on redraw.
       *  preRender - first class function executed before any redraw();
       *  postRender - first class function executed after any redraw();

    Inherits from: Nothing

    Main window superclass
*/
var Window = Class.extend({   

   constructor: function (x, y, w, h, windowstyle) {
        
        //Sub objects
        if (windowstyle) this.style = windowstyle;
          else this.style = new WindowStyle(WINDOWSTYLE_DEFAULT);
        

               
        //Sizing
        if (h) this.h = h; else this.h = 80;
        if (x) this.x = x; else this.x = 0;
        if (w) this.w = w; else this.w = GetScreenWidth() - this.x;
        if (y) this.y = y; else this.y = GetScreenHeight() - this.h; 
    },

    //Note properties are defined in constructor for convenience
    //These properties below are unneccessary but included for clarity
    x:undefined, 
    y:undefined, 
    w:undefined, 
    h:undefined, 
    style:undefined, 

    preRender: undefined, //type: first class function
    postRender: undefined,
    //These properties are less commonly modified aspects of the window.
    margin: 14,
    introAnim: WINDOWANIM_NONE,
    outtroAnim: WINDOWANIM_NONE,
    visible: false,
    animSpeed: 65, //fps 
    animInc: 9, //rate of change
    opacity: 255,
    background: BACKGROUND_NONE,  //rendermap OR an image to blit as background OR none
    flipScreen: false,
    
    /* METHOD: runAnim
    
       Parameters:
         * animName (string) - Which animation to run.
         * reverse (bool) - set to true when outtro anim is being played
       
       Runs a window animation.
    
    */
    runAnim: function (animName, reverse) {
        
        
            //GROW ANIMATION
            if (animName==WINDOWANIM_GROW) {
        
                var animDone = false;
            
                var centerX = Math.floor(this.x + (this.w / 2))
                var centerY = Math.floor(this.y + (this.h / 2))
                SetFrameRate(this.animSpeed);
                if (reverse) {
                    var currentWidth = this.w - (this.margin * 2)
                    var currentHeight = this.h - (this.margin * 2)
                    while (!animDone) {
                        this.drawBackground();
                        if (this.preRender) this.preRender();
                        if (currentHeight > ((this.margin ))) currentHeight -= this.animInc; else currentHeight = this.margin ;
                        if (currentWidth > ((this.margin ))) currentWidth -= this.animInc; else currentWidth = this.margin ;
                    
                        if (this.style != WINDOWSTYLE_NONE) this.style.drawWindow(centerX - (currentWidth /2),centerY - (currentHeight /2),currentWidth,currentHeight);
                        if (this.postRender) this.postRender();
                        FlipScreen();
                        if ((currentWidth <= (this.margin )) && (currentHeight <= (this.margin ))) {
                            animDone = true;
                        }
                    }
                }
                else {
                    var currentWidth = this.margin;
                    var currentHeight = this.margin ;
                    while (!animDone) {
                        this.drawBackground();
                        if (this.preRender) this.preRender();
                        if (currentHeight < (this.h - (this.margin *2))) currentHeight += this.animInc; else currentHeight = this.h - (this.margin *2);
                        if (currentWidth < (this.w - (this.margin *2))) currentWidth += this.animInc; else currentWidth = this.w - (this.margin *2);

                        if (this.style != WINDOWSTYLE_NONE) this.style.drawWindow(centerX - (currentWidth /2),centerY - (currentHeight /2),currentWidth,currentHeight);
                        if (this.postRender) this.postRender();
                        FlipScreen();
                        if ((currentWidth >= (this.w - (this.margin * 2))) && (currentHeight>= (this.h - (this.margin * 2)))) {
                            animDone = true;
                        }
                    }
                }
            }
        
            
            //FADE ANIMATION
            if (animName==WINDOWANIM_FADE) {
                
                var maxOpacity = this.opacity
                var animDone = false;
                SetFrameRate(this.animSpeed);
                if (reverse) this.opacity = maxOpacity; else this.opacity = 0;
                while (!animDone) {
                
                    if (!reverse) {
                        this.opacity += this.animInc;
                    
                        if (this.opacity >= maxOpacity) {
                          this.opacity = maxOpacity;
                          animDone = true;
                        }
                        
                    }
                    else {
                       this.opacity -= this.animInc;
                       if (this.opacity <= 0) {
                       this.opacity = 0;
                       animDone = true;
                       }
                   
                    }
                this.drawBackground();
                
                if (this.preRender) this.preRender();
                this.redraw();
                if (this.postRender) this.postRender();
                FlipScreen();
                
                
            }
            
        }
        
    },
    
    /* METHOD: initialize
    
       Parameters: None
       
       Sets up the window for showing.
    
    */      
    initialize: function () {
        if (this.opacity == 0) {this.opacity = 255;
        if (this.style != WINDOWSTYLE_NONE) this.style.setColorMask(new Color(this.style.getColorMask().getRed(),this.style.getColorMask().getGreen(),this.style.getColorMask().getBlue(),255));}
    
    },
    
    /* METHOD: Show
       
       Runs the intro anim, hides the window. Does not dispose it.
    
    */    
    show: function () {

        this.visible = true;
        
        if (this.flipScreen) {
            this.initialize();
            
            this.runAnim(this.introAnim,false);
        
            
                
        }
        this.update();
    },
    
    
    /* METHOD: Hide
    
       Parameters:
         * outtroAnim - set the outtroAnim
       
       Runs the outro anim, hides the window. Does not dispose it.
    
    */
    hide: function () {
        
        if (this.flipScreen) {   
            this.runAnim(this.outtroAnim,true); 
        }
        this.drawBackground();
        if (this.flipScreen) FlipScreen();
        this.visible = false;        
    },
    
    /* METHOD: Redraw
    
       Parameters: None

       Performs drawing tasks, no FlipScreen(); 
    */
    redraw: function () {
    
        if (!this.visible) return;
        if (this.style != WINDOWSTYLE_NONE) if (this.style.getColorMask().alpha != this.opacity) {
               this.style.setColorMask(new Color(this.style.getColorMask().getRed(), this.style.getColorMask().getGreen(), this.style.getColorMask().getBlue(), this.opacity));
        }
        
         
             if (this.style != WINDOWSTYLE_NONE) this.style.drawWindow(this.x + this.margin,this.y + this.margin,this.w - 2 * this.margin,this.h - 2 * this.margin);
          
        
    },
    
    
    drawBackground: function () {
        if (this.background == BACKGROUND_RENDERMAP && IsMapEngineRunning()) {
            RenderMap();
        }
        else if (this.background == BACKGROUND_RENDERMAP && !IsMapEngineRunning()) {
            //Error message here?? maybe...
        }
        else if (this.background != BACKGROUND_NONE) {
            this.background.blit(0,0);
        }
    },
    
    
    /* METHOD: Update
    
       Parameters: None
         

       Renders BG. Redraws. Flips Screen.
    */    
    update: function () {
        this.drawBackground();
        if (this.preRender) this.preRender();
        this.redraw();
        if (this.postRender) this.postRender();
        if (this.flipScreen) {
        
            FlipScreen();
        }
    }
});


 /* OBJECT: ImageWindow 
    Parameters: 
       * windowstyle, x, y, w, h - inherited from superclass
       * image - Image or surface object to use.
       * imageAlignment - (use constants) to define the image alignment in the window.
       * imageOpacity - Opacity of the image

    Non Parametised Properties
       * margin, opacity, preRender,postRender,background, visible, animInc, animSpeed, introAnim, outtroAnim - All inherited from superclass.
       * imageVisible - Whether to draw the image.
       * imageLeftPadding, imageRightPadding, imageTopPadding, imageBottomPadding - Padding around the image.

    Inherits from: Window

    A Window that contains either an image OR a surface.
*/
 var ImageWindow = Window.extend({
 
 
    constructor: function (image, imageAlignment, x, y, w, h, windowstyle) {
        
        //To build on existing constructor, call the superclass.
        this.base(x,y,w,h,windowstyle); //.call(this);
        if (imageAlignment) this.imageAlignment = imageAlignment;
          
        if (image.setAlpha || image.toSurface) this.image = image; 
          else Abort("You must provide an image or surface to ImageWindow");      
        
        
    },
    
    imageAlignment: ALIGNMENT_TOPLEFT,
    
    image:undefined,
    imageOpacity:255,
    imageVisible:true,
    
    imageLeftPadding:5,
    imageRightPadding:5,
    imageTopPadding:5,
    imageBottomPadding:5,
    
    /* METHOD: Redraw
    
       Parameters: None

       Performs drawing tasks, inherited from superclass. 
    */    
    redraw: function () {
        this.base();
        var x;
        var y;
        if (this.imageAlignment == ALIGNMENT_TOPLEFT || this.imageAlignment == ALIGNMENT_MIDDLELEFT || this.imageAlignment == ALIGNMENT_BOTTOMLEFT) {
            x = this.x + this.margin + this.imageLeftPadding;
        }
        else if (this.imageAlignment == ALIGNMENT_TOPRIGHT || this.imageAlignment == ALIGNMENT_MIDDLERIGHT || this.imageAlignment == ALIGNMENT_BOTTOMRIGHT) {
            x = this.x + this.w - this.margin - this.imageRightPadding - this.image.getWidth();
        }      
        else {
            x = Math.floor(this.x + (this.w / 2 ) - (this.image.getWidth() / 2));
        }
        if (this.imageAlignment == ALIGNMENT_TOPLEFT || this.imageAlignment == ALIGNMENT_TOPRIGHT || this.imageAlignment == ALIGNMENT_TOPCENTER) {
            y = this.y + this.margin + this.imageTopPadding;
        }
        else if (this.imageAlignment == ALIGNMENT_MIDDLELEFT || this.imageAlignment == ALIGNMENT_MIDDLERIGHT || this.imageAlignment == ALIGNMENT_MIDDLECENTER) {
            y = Math.floor(this.y + (this.h / 2) - (this.image.getHeight() / 2));
        }
        else {
            y = this.y + this.h - this.margin - this.imageBottomPadding - this.image.getHeight();
        }
        //polymorphism at work
        
        if (this.image.toImage) { //i.e we have a surface
        try { this.image.setAlpha(this.opacity * (this.imageOpacity /255));} catch (ex) {}
            
            
        
        }
        else if (this.image.toSurface) { //i.e we have an image

             this.image.mask = new Color(this.image.mask.getRed(),this.image.mask.getGreen(),this.image.mask.getBlue(),this.opacity* (this.imageOpacity /255));
        
        }
        
        if (this.imageVisible) this.image.blit(x,y);
    }
 
 });
 
 


 

/* OBJECT: TextWindow 
    Parameters: 
       * x, y, w, h, windowstyle - As in Window Superclass
       * font - Font object for text
       * text (string) - Text to display. 
       
    Non-parametised Properties:
       * margin, opacity, animSpeed, background, animInc, outtroAnim, introAnim, visible,preRender,postRender - As in superclass
       * textOpacity - Opacity of text
       * textAnimSpeed - Speed of text animation
       * textAnimInc - Rate of animation change.
       * textOffset - Offset of textbox. For use mostly in animation.
       * textVisible - Whether or not to redraw text on redraw.
       * textLeftPadding, textRightPadding, textTopPadding, textBottomPadding - default 5
       
    Inherits from: Window

    An animating, text-displaying window.
*/

var TextWindow = Window.extend({ 

    constructor: function (text, x, y, w, h,font, windowstyle) {
        
        //To build on existing constructor, call the superclass.
        this.base(x,y,w,h,windowstyle); //.call(this);
        if (font) this.font = font;
          else this.font = new Font();       
        if (text) this.text = text; 
          else this.text = "";      
          
        
    },
          
    textLeftPadding: 5,
    textRightPadding: 5,
    textTopPadding: 5,
    textBottomPadding: 5,
    introText: TEXTANIM_NONE,
    outtroText: TEXTANIM_NONE,
    textOpacity: 255,
    textAnimSpeed: 65,
    textAnimInc: 25,
    textVisible: true,
    textOffset: 0,
    //Just a reminder that this property exists, but is defined by constructor.
    text: undefined,
    font:undefined,    
    wrapping:true,

    /* METHOD: initializeText
    
       Parameters: None
       
       Prepares the properties for specific types of animation before window is shown.
    */
    initializeText: function () {
        if (this.textOpacity == 0) this.textOpacity = 255;
        this.textOffset = 0;
        
        if (this.introText ==  TEXTANIM_FADE) this.textOpacity=0;
        if (this.introText ==  TEXTANIM_CHARBYCHAR)  this.textVisible=false;
        if (this.introText ==  TEXTANIM_SLIDE) this.textOffset=this.h +15;
    
    },
    
    
    /* METHOD: show - inherited from superclass
    
       Parameters: none
       
       An extension to setup states and show text
    */
    show: function() {
    
        this.initializeText()
        // call the superclass
        
        this.base();
        
        this.showText();
        
    },
    
    
    /* METHOD: Hide - Inherited from superclass
    
       Parameters: none
          
          
       Hides the window. An extension to add hideText.
     */
    hide: function() {
        this.hideText();
        this.base();
    },
    

    
    /* METHOD: runTextAnim
    
       Parameters:
          * animName - name of the text animation to run
          * reverse - set to true for outtros
          
       Runs text animations through and flips() them.
     */         
    runTextAnim: function(animName, reverse) {
        var no_wrapping = !this.wrapping

        //FADE ANIMATION
        if (animName==TEXTANIM_FADE || (animName==TEXTANIM_CHARBYCHAR && reverse)) {
        
            maxOpacity=this.opacity;
            var animDone = false;
            SetFrameRate(this.textAnimSpeed);

            if (reverse) this.textOpacity = maxOpacity; else this.textOpacity = 0;
            while (!animDone) {
                if (!reverse) {
                    
                    this.textOpacity += this.textAnimInc;
                    
                    if (this.textOpacity >= maxOpacity) {
                      this.textOpacity = maxOpacity;
                      animDone = true;
                    }
                    
                }
                else {
                    this.textOpacity -= this.textAnimInc;
                    if (this.textOpacity <= 0) {
                      this.textOpacity = 0;
                      animDone = true;
                    }
                    
                }
                this.drawBackground();
                if (this.preRender) this.preRender();
                this.redraw();
                if (this.postRender) this.postRender();
                FlipScreen();
            }
        }
        
        //SLIDE ANIMATION
        if (animName==TEXTANIM_SLIDE) {
            
            if (!reverse) baseOffset = this.textOffset - (this.h +15); else baseOffset = this.textOffset;
            //Seeing as this increments alot faster than the fade we will slow it down:
            textAnimInc = Math.floor(this.textAnimInc / 15) + 1
						
						//For a downtrack override
						

            if (reverse) this.textOffset = baseOffset; else this.textOffset = baseOffset + (this.h - this.margin * 2 - (this.textLeftPadding + this.textRightPadding));
            
            var animDone = false;
            SetFrameRate(this.textAnimSpeed);
            
            while (!animDone) {
                if (!reverse) {
                    
                    this.textOffset -= textAnimInc;
                    
                    if (this.textOffset <= baseOffset) {
                      this.textOffset = baseOffset;
                      animDone = true;
                    }
                    
                }
                else {
                    
                    this.textOffset -= textAnimInc;
                    if (this.textOffset <= baseOffset + -(this.h - this.margin * 2 - (this.textLeftPadding + this.textRightPadding)) ) {
                      
                      animDone = true;
                    }
                    
                }
                this.drawBackground();
                if (this.preRender) this.preRender();
                this.redraw();
                if (this.postRender) this.postRender();
                FlipScreen();
            }
        }
        
        //TYPEWRITER ANIMATION
        if (animName==TEXTANIM_CHARBYCHAR && !reverse) {
        
            this.textVisible = false;
            var x = this.x + this.textLeftPadding + this.margin;
            var y = this.y + this.textTopPadding + this.margin;
            var w = this.w - (this.margin * 2) - this.textLeftPadding - this.textRightPadding;
            
            var h = this.h - (this.margin * 2) - this.textTopPadding - this.textBottomPadding;
            if (no_wrapping) 
                var text =this.text
                
            else
                var text = this.text.wordWrapByFontWidth(this.font,w).stitch("\n");
            SetFrameRate(this.textAnimSpeed);
            for (var i = 0;i <= text.length;i++) {
            
                currentCharacter = text.charAt(i);

                this.drawBackground();
                if (this.preRender) this.preRender();
                this.redraw();
                if (this.postRender) this.postRender();
                this.font.drawTextBox(x,y,GetScreenWidth() *2,h,this.textOffset,text.substr(0,i));
                if (IsKeyPressed(Game.keyConfirm)) {
                    i = text.length;
                }
                if (this.nameWindow) this.nameWindow.redraw();                
                FlipScreen();

                  


            }
            this.textVisible = true;

            this.drawBackground();
            if (this.preRender) this.preRender();
            this.redraw();
            if (this.postRender) this.postRender();
            FlipScreen();
        }
        
        
    },
    
          
    /* METHOD: showText
    
       Parameters: none
         
       Shows text, as a subset of the Show method.
     */
    showText: function() {
				this.textVisible = true;
				if (this.flipScreen) {
            this.initializeText();
            this.runTextAnim(this.introText,false);   
        }
    },
    /* METHOD: hideText
    
       Parameters: none
          
       Hides text, as a subset of the Hide method.
     */     
    hideText: function(textAnim) {
        
        if (this.flipScreen) {
            if (textAnim) this.outtroText = textAnim;
            this.runTextAnim(this.outtroText,true);
        }
        this.textVisible = false; 
    },
    
    
    /* METHOD: redraw - Inherited from superclass
          
       Extends on Superclass Function. Adds Text Drawing to Update
     */  
     redraw: function() {
         this.base();
         
         if (this.font.getColorMask().getAlpha() != Math.floor((this.textOpacity / 255) * this.opacity)) {
             this.font.setColorMask(new Color(this.font.getColorMask().getRed(), this.font.getColorMask().getGreen(), this.font.getColorMask().getBlue(), Math.floor((this.textOpacity / 255) * this.opacity)));
         }
         
         
         if (this.textVisible) {
             if (!this.wrapping) 
                 this.font.drawTextBox(this.x + this.margin + this.textLeftPadding,this.y + this.margin + this.textTopPadding,GetScreenWidth() * 2,this.h - 2 * this.margin - (this.textTopPadding + this.textBottomPadding),this.textOffset,this.text);            
                 
                 
             else
                 this.font.drawTextBox(this.x + this.margin + this.textLeftPadding,this.y + this.margin + this.textTopPadding,GetScreenWidth() * 2,this.h - 2 * this.margin - (this.textTopPadding + this.textBottomPadding) ,this.textOffset,this.text.wordWrapByFontWidth(this.font,this.w - (this.margin) * 2 - this.textLeftPadding - this.textRightPadding).stitch("\n"));            
         }
         
     },
     

});

/* OBJECT: MessageWindow 
    Parameters: 
       * x, y, w, h, windowstyle, font - As in Superclass
       * text - broken into MessageArray by \n.
       * domessages - Immediately launch the window.
       
    Non-parametised Properties:
       * margin, opacity, animSpeed, background, animInc, outtroAnim, introAnim, visible, textOpacity,textAnimSpeed,textAnimInc,textOffset,textVisible,textPadding - As in superclass
       * showArrow - internal property to display arrow or not.
       
    Inherits from: TextWindow

    A window specifically for "Talking" dialogs..
*/

var MessageWindow = TextWindow.extend({ 

    constructor: function (text, name, portrait, domessages, x, y, w, h, font, windowstyle) {
        if (!name) name = "";
        
        //To build on existing constructor, call the superclass.
        
        this.base("",x,y,w,h,font,windowstyle); //.call(this);
        
        this.lineHeight = Math.floor((this.h - (this.margin) * 2 - (this.textTopPadding + this.textBottomPadding)) / this.font.getHeight());
        this.text = text;
        
//this.messageArray = new Array();        
        this.arrow = new Image(IMAGE_ARROW_DOWN);
        
        this.arrowX = Math.floor(this.w / 2) - Math.floor(this.arrow.getWidth() /2);
        this.arrowY = this.h - this.arrow.getHeight();
        
        this.nameWindow = new TextWindow();
        if (name) this.nameWindow.text = name;
        if (portrait) this.portraitWindow = new ImageWindow(portrait); else this.portraitWindow = IMAGE_NONE;
        
        if (domessages) this.doMessages(text,name);
        
    },
    
    //Override. Resetting default to "rendermap" as this is significantly more commonly used in such a context.
    background: BACKGROUND_RENDERMAP,
    //More overrides
    introAnim: WINDOWANIM_FADE,
    outtroAnim: WINDOWANIM_FADE,
    introText: TEXTANIM_CHARBYCHAR,
    outtroText: TEXTANIM_FADE,
    flipScreen: true,
    //Amount of lines at a time to display.
    lineHeight: undefined,
    
    messageArray: undefined,
    
    arrow: undefined,
    
    arrowX: undefined,
    
    arrowY: undefined,
    
    arrowVisible: false,
    
    portraitWindow: undefined,
    
    portraitWindowAlignment: ALIGNMENT_MIDDLERIGHT,
    
    currentLine: 0,
    
    arrowOffset: 0,
    
    wrapping: false, //WE USE OUR OWN WRAPPING
    
    nameWindow: undefined,
    
    nameWindowAlignment: ALIGNMENT_TOPLEFT, 
    
    
    /* METHOD: InitializeWindows
    
       An internal method to configure window positions.
    */
    initializeWindows: function(name,portrait) {  
        if (portrait) this.portraitWindow = new ImageWindow(portrait,ALIGNMENT_MIDDLECENTER,false,false,false,false,this.portraitWindow.style);
        
        
        if (name) this.nameWindow.text = name;
        
        
        
        var x;
        var y;
        var w = this.font.getStringWidth(this.nameWindow.text) + this.margin * 2 ;
        var h = this.font.getHeight() + this.margin * 2;
        
        if (this.nameWindowAlignment == ALIGNMENT_TOPLEFT || this.nameWindowAlignment == ALIGNMENT_BOTTOMLEFT || this.nameWindowAlignment == ALIGNMENT_MIDDLELEFT) {
            x = this.x + this.margin + this.textLeftPadding;
            
          
        } 
        else if (this.nameWindowAlignment == ALIGNMENT_TOPCENTER || this.nameWindowAlignment == ALIGNMENT_BOTTOMCENTER || this.nameWindowAlignment == ALIGNMENT_MIDDLECENTER) {
            x = Math.floor(this.x + this.w / 2 - (w / 2)); 
        } 
        else {
            x = this.x + this.w - this.margin - this.textRightPadding - w;
            
            
        
        }
        
        if (this.nameWindowAlignment == ALIGNMENT_TOPLEFT || this.nameWindowAlignment == ALIGNMENT_TOPCENTER || this.nameWindowAlignment == ALIGNMENT_TOPRIGHT) {
            y = this.y + this.margin + this.textTopPadding - (this.font.getHeight()+ this.margin * 2)
            
        
        }
        else if (this.nameWindowAlignment == ALIGNMENT_BOTTOMLEFT || this.nameWindowAlignment == ALIGNMENT_BOTTOMCENTER || this.nameWindowAlignment == ALIGNMENT_BOTTOMRIGHT)  {
            y = this.y + this.h - this.margin - this.textBottomPadding - 5;
        }
        else {
            y = Math.floor(this.y + this.h / 2 - h / 2);
        }
        this.nameWindow = new TextWindow(this.nameWindow.text,x,y,w,h,this.nameWindow.font,this.nameWindow.style);
        this.nameWindow.textLeftPadding = 0;
        this.nameWindow.textTopPadding = 0;
        this.nameWindow.textRightPadding = 0;
        this.nameWindow.textBottomPadding = 0;
        

        if (this.portraitWindow != IMAGE_NONE) {

            x = -1;
            y = -1;
            w = this.portraitWindow.image.getWidth()  + this.portraitWindow.margin * 2;
            h = this.portraitWindow.image.getHeight()  + this.portraitWindow.margin * 2;
            
            if (this.portraitWindowAlignment == ALIGNMENT_TOPLEFT || this.portraitWindowAlignment == ALIGNMENT_BOTTOMLEFT || this.portraitWindowAlignment == ALIGNMENT_MIDDLELEFT) {
                x = this.x + this.margin + this.textLeftPadding;
              
                
            } 
            else if (this.portraitWindowAlignment == ALIGNMENT_TOPCENTER || this.portraitWindowAlignment == ALIGNMENT_BOTTOMCENTER || this.portraitWindowAlignment == ALIGNMENT_MIDDLECENTER) {
                x = Math.floor(this.x + this.w / 2 - (w / 2)); 
                
            } 
            else {
                x = this.x + this.w - this.margin - this.textRightPadding - w;
              
                
            }
        
            if (this.portraitWindowAlignment == ALIGNMENT_TOPLEFT || this.portraitWindowAlignment == ALIGNMENT_TOPCENTER || this.portraitWindowAlignment == ALIGNMENT_TOPRIGHT) {
                y = this.y + this.margin + this.textTopPadding - h;
        
            }
            else if (this.portraitWindowAlignment == ALIGNMENT_BOTTOMLEFT || this.portraitWindowAlignment == ALIGNMENT_BOTTOMCENTER || this.portraitWindowAlignment == ALIGNMENT_BOTTOMRIGHT)  {
                y = this.y + this.h - this.margin - this.textBottomPadding - 5;
            }
            else {
                y = Math.floor(this.y + this.h / 2 - h / 2);
            }
        
            this.portraitWindow.imageLeftPadding = 0;
            this.portraitWindow.imageRightPadding = 0;
            this.portraitWindow.imageTopPadding = 0;
            this.portraitWindow.imageBottomPadding = 0;
            this.portraitWindow.x = x;
            this.portraitWindow.y = y;
            this.portraitWindow.w = w;
            this.portraitWindow.h = h;
        }
        
    
    },
    
    
    /* METHOD: doMessages 
          
       Parameters: text, name - as in description
       
       Shows if necessary, paginates and displays text, with namebox text Name, if specified, hides.
    */  
    doMessages: function(text,name, portrait) {
			
			  var oldpadding = this.textLeftPadding;
        var oldpadding2 = this.textRightPadding;
        
			  this.initializeWindows(name,portrait);
			  if (this.nameWindowAlignment == ALIGNMENT_MIDDLERIGHT) this.textRightPadding += this.textRightPadding + this.nameWindow.w;
			  if (this.nameWindowAlignment == ALIGNMENT_MIDDLELEFT) this.textLeftPadding += this.textLeftPadding + this.nameWindow.w;
			  
			  if (this.portraitWindowAlignment == ALIGNMENT_MIDDLELEFT && this.portraitWindow != IMAGE_NONE && oldpadding == this.textLeftPadding) this.textLeftPadding += this.textLeftPadding + this.portraitWindow.w;            
			  if (this.portraitWindowAlignment == ALIGNMENT_MIDDLERIGHT && this.portraitWindow != IMAGE_NONE  && oldpadding2 == this.textRightPadding) this.textRightPadding += this.textRightPadding + this.portraitWindow.w;            
				this.arrowX = Math.floor(this.w / 2) - Math.floor(this.arrow.getWidth() /2);
        this.arrowY = this.h - this.arrow.getHeight();
        this.currentLine = 0;
        if (!text) text = this.text;
        if (text) this.messageArray = text.wordWrapByFontWidth(this.font,this.w - (this.textLeftPadding + this.textRightPadding) - (this.margin * 2));
				this.lineHeight = Math.floor((this.h - (this.margin) * 2 - (this.textTopPadding + this.textBottomPadding)) / this.font.getHeight());

        this.text ="";
        
        if (!this.visible) this.show();
        
        

        this.arrowVisible = true;
        
        
        while (this.arrowVisible) {

            if (this.messageArray.length - this.currentLine <= this.lineHeight) {

                this.arrowVisible = false;   
            }
            this.hideText();
            this.text ="";
            
            for (var i = this.currentLine; i < this.currentLine + this.lineHeight;i++) {

                if (this.messageArray[i]) {

                   this.text = this.text  + this.messageArray[i] + '\n';
                    
                }
                
            }
            
            
            
            this.showText();
            SetFrameRate(12);
            while (AreKeysLeft()) GetKey();
            //Keyboard buffer Cleared.. so..
            var theKey = "";
            
            while (theKey != Game.keyConfirm) {
               while (AreKeysLeft()) theKey = GetKey(); 
               this.update();
            }
            while (AreKeysLeft()) GetKey();
            this.currentLine += this.lineHeight;
        }   
        this.hide();             
        
            this.textLeftPadding = oldpadding;
        
        
            this.textRightPadding = oldpadding2; 
        

    
    },
    
    /* METHOD: animateArrow 
     
       Animates the arrow in the redraw loop.
     */
    animateArrow: function() {
       
           if (this.arrowOffset == 2) 
               this.arrowOffset -= 2;
           else
               this.arrowOffset += 1;    
       
    
    
    
    },
    
    
    /* METHOD: redraw - Inherited from superclass
          
       Extends on Superclass Function. Adds arrow and name box to drawing update
     */      
    redraw: function() {  
    
        this.base();
         
        if (this.arrowVisible) {
            this.animateArrow();
            this.arrow.blit(this.x + this.arrowX, this.y + this.arrowY + this.arrowOffset);
        }
        if (this.portraitWindow != IMAGE_NONE) {
						
            this.portraitWindow.opacity = this.opacity;
            this.portraitWindow.visible = this.visible;
            this.portraitWindow.redraw();
        }
        if (this.nameWindow.text != "") {
        
            this.nameWindow.opacity = this.opacity;
            this.nameWindow.visible = this.visible;
            this.nameWindow.redraw();
        }
    },
    
    
    /* METHOD: runAnim - Inherited from superclass
          
       Extends on Superclass Function. Hides arrow while animating.
     */  
    runAnim: function(animName,reverse) {
        var tempArrowVisible = this.arrowVisible;
        
        this.arrowVisible = false;
        this.base(animName,reverse); 
        this.arrowVisible = tempArrowVisible;
    
    },
    
    runTextAnim: function(animName,reverse) {
        var tempArrowVisible = this.arrowVisible;
        
        this.arrowVisible = false;
        this.base(animName,reverse); 
        this.arrowVisible = tempArrowVisible;
    
    }
    
});
